/*
** Program: Clean Prover System
** Module:  PolishRead (.icl)
** 
** Author:  Maarten de Mol
** Created: 26 March 2001
*/

implementation module
	PolishRead

import
	StdEnv,
	StdIO,
	Errors,
	States


// -------------------------------------------------------------------------------------------------------------------------------------------------
class StateMonad state
// -------------------------------------------------------------------------------------------------------------------------------------------------
where
	ret							:: !a !*state -> (!Error, !a, !*state)
	(>>=) infixl				:: !(StateM a state) !(a -> StateM b state) !*state -> (!Error, !b, !*state)
:: StateM a state :== a *state -> (Error, a, *state)

// -------------------------------------------------------------------------------------------------------------------------------------------------
:: *FileState =
// -------------------------------------------------------------------------------------------------------------------------------------------------
	{ fsFile					:: !*File
	, fsLineNumber				:: !Int
	, fsCharNumber				:: !Int
	, fsCurrentLine				:: !String
	}

// -------------------------------------------------------------------------------------------------------------------------------------------------
:: *SectionState =
// -------------------------------------------------------------------------------------------------------------------------------------------------
	{ ssName					:: !String
	, ssPointer					:: !SectionPtr
	, ssTheorems				:: ![TheoremPtr]
	, ssInputFile				:: !Maybe *FileState
	, ssKnownExprVars			:: ![CExprVarPtr]
	, ssKnownPropVars			:: ![CPropVarPtr]
	, ssUsedSymbols				:: ![HeapPtr]
	, ssUsedTheorems			:: ![HeapPtr]
	}

/*
// -------------------------------------------------------------------------------------------------------------------------------------------------
:: FileM	:== *FileState -> (Error, *FileState)
:: FileMA a	:== *FileState -> (Error, a, *FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------

// -------------------------------------------------------------------------------------------------------------------------------------------------
isValidIdChar :: !Char -> Bool
// -------------------------------------------------------------------------------------------------------------------------------------------------
isValidIdChar c
	| isAlphanum c							= True
	| isMember c special_chars				= True
	= False
	where
		special_chars
			= ['~@#$%^?!+-*<>\/|&=:_`.']

// -------------------------------------------------------------------------------------------------------------------------------------------------
isValidNameChar :: !Char -> Bool
// -------------------------------------------------------------------------------------------------------------------------------------------------
isValidNameChar c
	| isAlphanum c							= True
	| isMember c special_chars				= True
	= False
	where
		special_chars
			= ['@_']















// -------------------------------------------------------------------------------------------------------------------------------------------------
newFileState :: !CName !*File !*PState -> *FileState
// -------------------------------------------------------------------------------------------------------------------------------------------------
newFileState name file pstate
	=	{ openedFile		= file
		, shortFileName		= name
		, lineNumber		= 0
		, charNumber		= 0
		, currentLine		= "\n"
		, pstate			= pstate
		, exprVars			= []
		, propVars			= []
		, proof				= EmptyProof
		, knownSymbols		= []
		, knownTheorems		= []
		}

// -------------------------------------------------------------------------------------------------------------------------------------------------
disposeFileState :: !*FileState -> *PState
// -------------------------------------------------------------------------------------------------------------------------------------------------
disposeFileState fs=:{openedFile, pstate}
	# (error, pstate)						= accFiles (fclose openedFile) pstate
	= pstate

// -------------------------------------------------------------------------------------------------------------------------------------------------
setDepends :: ![HeapPtr] ![TheoremPtr] !*FileState -> *FileState
// -------------------------------------------------------------------------------------------------------------------------------------------------
setDepends symbols theorems fs=:{knownSymbols, knownTheorems}
	= {fs	& knownSymbols		= knownSymbols ++ symbols
			, knownTheorems		= knownTheorems ++ theorems
	  }

// -------------------------------------------------------------------------------------------------------------------------------------------------
appPState :: !(*PState -> *PState) !*FileState -> *FileState
// -------------------------------------------------------------------------------------------------------------------------------------------------
appPState f fs=:{pstate}
	# pstate								= f pstate
	= {fs & pstate = pstate}

// -------------------------------------------------------------------------------------------------------------------------------------------------
accPState :: !(*PState -> (a, *PState)) !*FileState -> (!a, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
accPState f fs=:{pstate}
	# (a, pstate)							= f pstate
	= (a, {fs & pstate = pstate})

// -------------------------------------------------------------------------------------------------------------------------------------------------
acc2PState :: !(*PState -> (a, b, *PState)) !*FileState -> (!a, !b, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
acc2PState f fs=:{pstate}
	# (a, b, pstate)						= f pstate
	= (a, b, {fs & pstate = pstate})















// -------------------------------------------------------------------------------------------------------------------------------------------------
(>>>) infixr 5 :: FileM (FileMA a) -> FileMA a | DummyValue a
// -------------------------------------------------------------------------------------------------------------------------------------------------
(>>>) action1 action2
	= act action1 action2
	where
		act :: !FileM !(FileMA a) !*FileState -> (!Error, !a, !*FileState) | DummyValue a
		act action1 action2 fs
			# fs							= eatSpaces fs
			# (error, fs)					= action1 fs
			| isError error					= (error, DummyValue, fs)
			# fs							= eatSpaces fs
			= action2 fs

// -------------------------------------------------------------------------------------------------------------------------------------------------
(>>=) infixl 6 :: (FileMA a) (a -> FileMA b) -> FileMA b | DummyValue b
// -------------------------------------------------------------------------------------------------------------------------------------------------
(>>=) action1 mk_action2
	= act action1 mk_action2
	where
		act :: !(FileMA a) !(a -> FileMA b) !*FileState -> (!Error, !b, !*FileState) | DummyValue b
		act action1 mk_action2 fs
			# fs							= eatSpaces fs
			# (error, a, fs)				= action1 fs
			| isError error					= (error, DummyValue, fs)
			# fs							= eatSpaces fs
			= mk_action2 a fs

// -------------------------------------------------------------------------------------------------------------------------------------------------
output :: !a -> FileMA a
// -------------------------------------------------------------------------------------------------------------------------------------------------
output value
	= \fs -> (OK, value, fs)

// -------------------------------------------------------------------------------------------------------------------------------------------------
parseError :: !String -> FileMA a | DummyValue a
// -------------------------------------------------------------------------------------------------------------------------------------------------
parseError msg
	= report_error
	where
		report_error :: !*FileState -> (!Error, !a, !*FileState) | DummyValue a
		report_error fs=:{shortFileName, lineNumber, charNumber}
			# error							= [X_ParseSection shortFileName lineNumber charNumber msg]
			= (error, DummyValue, fs)













// -------------------------------------------------------------------------------------------------------------------------------------------------
advanceLine :: !*FileState -> (!Error, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
advanceLine fs=:{openedFile, shortFileName, lineNumber, charNumber, currentLine}
	| currentLine <> "\n"
		# error								= [X_ParseSection shortFileName lineNumber charNumber "End of line expected."]
		= (error, fs)
	# (ended, file)							= fend openedFile
	| ended
		# error								= [X_ParseSection shortFileName lineNumber charNumber "Unexpected end of file."]
		= (error, {fs & openedFile = file})
	# (line, file)							= freadline file
	# fs									= {fs	& openedFile	= file
													, lineNumber		= lineNumber + 1
													, charNumber		= 1
													, currentLine		= line
											  }
	= (OK, fs)

// -------------------------------------------------------------------------------------------------------------------------------------------------
eatSpaces :: !*FileState -> *FileState
// -------------------------------------------------------------------------------------------------------------------------------------------------
eatSpaces fs=:{charNumber, currentLine}
	# line_size								= size currentLine
	# index									= skip_spaces 0 line_size currentLine
	# fs									= {fs	& charNumber	= charNumber + index
													, currentLine	= currentLine % (index, line_size-1)
											  }
	= fs
	where
		skip_spaces :: !Int !Int !String -> !Int
		skip_spaces index max text
			| index >= max					= index
			| text.[index] <> ' '			= index
			= skip_spaces (index+1) max text

// -------------------------------------------------------------------------------------------------------------------------------------------------
expectToken :: !String !*FileState -> (!Error, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
expectToken token fs=:{shortFileName, lineNumber, charNumber, currentLine}
	# size_token							= size token
	# size_line								= size currentLine
	| token <> currentLine % (0, size_token-1)
		# error								= [X_ParseSection shortFileName lineNumber charNumber ("Expected token " +++ token +++ ".")]
		= (error, fs)
	# fs									= {fs	& charNumber	= charNumber + size_token
													, currentLine	= currentLine % (size_token, size_line-1)
											  }
	= (OK, fs)

// -------------------------------------------------------------------------------------------------------------------------------------------------
lookAhead :: ![(String, FileMA a)] !(FileMA a) !*FileState -> (!Error, !a, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
lookAhead [] action fs
	= action fs
lookAhead [(text,act):alts] action fs=:{currentLine}
	# size_text								= size text
	| currentLine%(0, size_text-1) == text	= act fs
	= lookAhead alts action fs

// -------------------------------------------------------------------------------------------------------------------------------------------------
parseIdentifier :: !*FileState -> (!Error, !String, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
parseIdentifier fs
	= parseSatisfying "Identifier" isValidIdChar fs

// -------------------------------------------------------------------------------------------------------------------------------------------------
parseName :: !String !*FileState -> (!Error, !String, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
parseName what fs
	# what									= if (what == "") "Name" what
	= parseSatisfying what isValidNameChar fs

// -------------------------------------------------------------------------------------------------------------------------------------------------
parseNumber :: !*FileState -> (!Error, !Int, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
parseNumber fs=:{currentLine, charNumber}
	# size_line								= size currentLine
	| size_line > 0 && currentLine.[0] == '-'
		# fs								= {fs	& charNumber	= charNumber + 1
													, currentLine	= currentLine % (1, size_line-1)
											  }
		# (error, text, fs)					= parseSatisfying "Number" isDigit fs
		| isError error						= (error, DummyValue, fs)
		= (OK, ~(toInt text), fs)
	# (error, text, fs)						= parseSatisfying "Number" isDigit fs
	| isError error							= (error, DummyValue, fs)
	= (OK, toInt text, fs)

// -------------------------------------------------------------------------------------------------------------------------------------------------
parseSatisfying :: !String !(Char -> Bool) !*FileState -> (!Error, !String, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
parseSatisfying what valid fs=:{shortFileName, lineNumber, charNumber, currentLine}
	# size_line								= size currentLine
	# first_separator_index					= find_separator 0 size_line currentLine
	| first_separator_index < 1
		# error								= [X_ParseSection shortFileName lineNumber charNumber (what +++ "expected " +++ ".")]
		= (error, "", fs)
	# identifier							= currentLine % (0, first_separator_index - 1)
	# fs									= {fs	& charNumber	= charNumber + first_separator_index
													, currentLine	= currentLine % (first_separator_index, size_line-1)
											  }
	= (OK, identifier, fs)
	where
		find_separator :: !Int !Int !String -> Int
		find_separator index max text
			| index >= max					= -1
			| valid text.[index]			= find_separator (index+1) max text
			= index

// -------------------------------------------------------------------------------------------------------------------------------------------------
parseString :: FileMA String
// -------------------------------------------------------------------------------------------------------------------------------------------------
parseString
	=	expectToken					"\""						>>>
		parseSatisfying "end of string character" not_close		>>= \text ->
		expectToken					"\""						>>>
		output text
	where
		not_close :: !Char -> Bool
		not_close c
			= c <> '"'

// -------------------------------------------------------------------------------------------------------------------------------------------------
repeatedParse :: !Int !(FileMA a) -> FileMA [a]
// -------------------------------------------------------------------------------------------------------------------------------------------------
repeatedParse 0 parser
	=	output []
repeatedParse n parser
	=	parser													>>= \x ->
		repeatedParse (n-1) parser								>>= \xs ->
		output [x:xs]













// -------------------------------------------------------------------------------------------------------------------------------------------------
findName :: !CName ![Ptr a] !*CHeaps -> (!Bool, !Ptr a, !*CHeaps) | Pointer a
// -------------------------------------------------------------------------------------------------------------------------------------------------
findName name [ptr:ptrs] heaps
	# (ptr_name, heaps)						= getPointerName ptr heaps
	| name == ptr_name						= (True, ptr, heaps)
	= findName name ptrs heaps
findName name [] heaps
	= (False, nilPtr, heaps)

// -------------------------------------------------------------------------------------------------------------------------------------------------
newExprVars :: ![CName] !*FileState -> (!Error, ![CExprVarPtr], !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
newExprVars names fs=:{pstate, exprVars}
	# vars									= [{DummyValue & evarName = name} \\ name <- names]
	# (ptrs, pstate)						= accHeaps (newPointers vars) pstate
	# fs									= {fs	& pstate		= pstate
													, exprVars		= ptrs ++ exprVars
											  }
	= (OK, ptrs, fs)

// -------------------------------------------------------------------------------------------------------------------------------------------------
newPropVars :: ![CName] !*FileState -> (!Error, ![CPropVarPtr], !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
newPropVars names fs=:{pstate, propVars}
	# vars									= [{DummyValue & pvarName = name} \\ name <- names]
	# (ptrs, pstate)						= accHeaps (newPointers vars) pstate
	# fs									= {fs	& pstate		= pstate
													, propVars		= ptrs ++ propVars
											  }
	= (OK, ptrs, fs)

// -------------------------------------------------------------------------------------------------------------------------------------------------
disposeExprVars :: !Int !*FileState -> (!Error, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
disposeExprVars nr fs=:{exprVars}
	# fs									= {fs	& exprVars		= drop nr exprVars}
	= (OK, fs)

// -------------------------------------------------------------------------------------------------------------------------------------------------
disposePropVars :: !Int !*FileState -> (!Error, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
disposePropVars nr fs=:{propVars}
	# fs									= {fs	& propVars		= drop nr propVars}
	= (OK, fs)

// -------------------------------------------------------------------------------------------------------------------------------------------------
lookupExprVar :: !CName !*FileState -> (!Error, !CExprVarPtr, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
lookupExprVar name fs=:{shortFileName, lineNumber, pstate, exprVars, proof}
	# evars									= exprVars ++ proof.pCurrentGoal.glExprVars
	# (found, ptr, pstate)					= acc2Heaps (findName name evars) pstate
	# fs									= {fs	& pstate = pstate}
	| not found								= ([X_UnknownExprVar shortFileName lineNumber name], nilPtr, fs)
	= (OK, ptr, fs)

// -------------------------------------------------------------------------------------------------------------------------------------------------
lookupPropVar :: !CName !*FileState -> (!Error, !CPropVarPtr, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
lookupPropVar name fs=:{shortFileName, lineNumber, pstate, propVars, proof}
	# pvars									= propVars ++ proof.pCurrentGoal.glPropVars
	# (found, ptr, pstate)					= acc2Heaps (findName name pvars) pstate
	# fs									= {fs	& pstate = pstate}
	| not found								= ([X_UnknownPropVar shortFileName lineNumber name], nilPtr, fs)
	= (OK, ptr, fs)
















// -------------------------------------------------------------------------------------------------------------------------------------------------
newTheorem :: !CName !CPropH !*FileState -> (!Error, !*FileState)
// -------------------------------------------------------------------------------------------------------------------------------------------------
newTheorem name initial fs=:{pstate, knownTheorems}
	# goal									= {DummyValue & glToProve = initial}
	# (leaf, pstate)						= accHeaps (newPointer (ProofLeaf goal)) pstate
	# proof									=	{ pTree				= leaf
												, pLeafs			= [leaf]
												, pCurrentLeaf		= leaf
												, pCurrentGoal		= goal
												, pUsedTheorems		= []
												, pUsedSymbols		= [] // BEZIG
												}
	# theorem								= 	{ thName			= name
												, thInitial			= initial
												, thInitialText		= undef // BEZIG
												, thProof			= proof
												, thSection			= undef // BEZIG
												, thSubgoals		= True
												}
	# (ptr, pstate)							= accHeaps (newPointer theorem) pstate
	# fs									= {fs	& pstate		= pstate
													, knownTheorems	= [ptr:knownTheorems]
											  }
	= (OK, fs)
*/